<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Product extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'sku',
        'product_code',
        'manufacturer',
        'supplier',
        'category_id',
        'sub_category_id',
        'brand_id',
        'unit_id',
        'expiry_date',
        'quantity',
    ];

    protected $casts = [
        'expiry_date' => 'date:Y-m-d', // Ensures proper date handling
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // ✅ Define Relationships
    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function subCategory()
    {
        return $this->belongsTo(Category::class, 'sub_category_id');
    }

    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

   // ✅ If a product belongs to ONE unit:
    public function unit()
    {
        return $this->belongsTo(Unit::class, 'unit_id');
    }

    // ✅ If a product can have MULTIPLE units (e.g., different packaging options):
    public function units()
    {
        return $this->belongsToMany(Unit::class, 'product_units', 'product_id', 'unit_id');
    }

    public function images()
    {
        return $this->hasMany(ProductImage::class);
    }

    // ✅ Rename from productVariants() to variants()
    public function variants()
    {
        return $this->hasMany(ProductVariant::class);
    }

    // ✅ Auto-generate SKU and Product Code on creation
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($product) {
            $product->product_code = strtoupper(uniqid('PRD'));
            $namePart = substr(strtoupper(str_replace(' ', '', $product->name)), 0, 3);
            $manufacturerPart = substr(strtoupper(str_replace(' ', '', $product->manufacturer ?? 'GEN')), 0, 3);
            $randomDigits = mt_rand(1000, 9999);
            $product->sku = "{$namePart}-{$manufacturerPart}-{$randomDigits}";
            if (empty($product->expiry_date)) {
                $product->expiry_date = '9999-12-31'; // Default far future date
            }
        });
    }
}
