<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Category extends Model
{
    use HasFactory;

    protected $fillable = ['name', 'description', 'image', 'category_number', 'parent_id'];

    /**
     * ✅ Define Self-Referencing Relationship for Parent-Child Categories
     */
    public function parentCategory()
    {
        return $this->belongsTo(Category::class, 'parent_id');
    }

    public function subcategories()
    {
        return $this->hasMany(Category::class, 'parent_id');
    }

    public function products()
    {
        return $this->hasMany(Product::class);
    }

    /**
     * ✅ Auto-generate category_number before saving
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($category) {
            // If parent_id exists, generate subcategory number
            if ($category->parent_id) {
                $parentCategory = Category::find($category->parent_id);
                if ($parentCategory) {
                    $lastSubcategory = Category::where('parent_id', $category->parent_id)
                        ->orderBy('category_number', 'desc')
                        ->first();

                    $subCategoryNumber = $lastSubcategory
                        ? str_pad((int) substr($lastSubcategory->category_number, -2) + 1, 2, '0', STR_PAD_LEFT)
                        : '01';

                    $category->category_number = $parentCategory->category_number . $subCategoryNumber; // e.g., "01" -> "0101"
                }
            } else {
                // Generate category_number for parent category
                $lastCategory = Category::whereNull('parent_id')
                    ->orderBy('category_number', 'desc')
                    ->first();

                $category->category_number = $lastCategory
                    ? str_pad((int) $lastCategory->category_number + 1, 2, '0', STR_PAD_LEFT)
                    : '01';
            }
        });
    }
}
