<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Variant;
use Illuminate\Database\QueryException;
use Illuminate\Validation\ValidationException;
use Exception;

class VariantController extends Controller
{
    // Fetch all available variants
    public function index()
    {
        try {
            $variants = Variant::all();
            return response()->json(['status' => 'success', 'data' => $variants], 200);
        } catch (Exception $e) {
            \Log::error('Error fetching variants: ' . $e->getMessage());
            return response()->json(['status' => 'error', 'message' => 'Failed to fetch variants'], 500);
        }
    }

    // Create a new variant
    public function store(Request $request)
{
    try {
        $request->validate([
            'name' => 'required|string|max:255|unique:variants,name',
            'value' => 'required|string|max:255',
        ]);

        $variant = Variant::create([
            'name' => $request->name,
            'value' => $request->value,
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Variant created successfully',
            'data' => $variant
        ], 201);

    } catch (ValidationException $e) {
        return response()->json(['status' => 'error', 'message' => $e->errors()], 422);
    } catch (QueryException $e) {
        return response()->json(['status' => 'error', 'message' => 'Database error while creating variant'], 500);
    } catch (Exception $e) {
        \Log::error('Error creating variant: ' . $e->getMessage());
        return response()->json(['status' => 'error', 'message' => 'An unexpected error occurred'], 500);
    }
}


    // Get a specific variant by ID
    public function show($id)
    {
        try {
            $variant = Variant::findOrFail($id);
            return response()->json(['status' => 'success', 'data' => $variant], 200);
        } catch (Exception $e) {
            return response()->json(['status' => 'error', 'message' => 'Variant not found'], 404);
        }
    }

    // Update a variant
    public function update(Request $request, $id)
    {
        try {
            $variant = Variant::findOrFail($id);

            $request->validate([
                'name' => 'required|string|max:255',
                'value' => 'required|string|max:255',
            ]);

            $variant->update($request->all());

            return response()->json(['status' => 'success', 'message' => 'Variant updated successfully', 'data' => $variant], 200);
        } catch (ValidationException $e) {
            return response()->json(['status' => 'error', 'message' => $e->errors()], 422);
        } catch (Exception $e) {
            return response()->json(['status' => 'error', 'message' => 'Variant not found or update failed'], 404);
        }
    }

    // Delete a variant
    public function destroy($id)
    {
        try {
            $variant = Variant::findOrFail($id);
            $variant->delete();
            return response()->json(['status' => 'success', 'message' => 'Variant deleted successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['status' => 'error', 'message' => 'Variant not found'], 404);
        }
    }
}
