<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Unit;
use Illuminate\Database\QueryException;
use Illuminate\Validation\ValidationException;
use Exception;

class UnitController extends Controller
{
    // Fetch all units
    public function index()
    {
        try {
            $units = Unit::all();
            return response()->json(['status' => 'success', 'data' => $units], 200);
        } catch (Exception $e) {
            \Log::error('Error fetching units: ' . $e->getMessage());
            return response()->json(['status' => 'error', 'message' => 'Failed to fetch units'], 500);
        }
    }

    // Create a new unit
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255|unique:units,name',
                'shortform' => 'required|string|max:10|unique:units,shortform', // ✅ Fix field name
            ]);

            $unit = Unit::create([
                'name' => $request->name,
                'shortform' => $request->shortform, // ✅ Fix field name
            ]);

            return response()->json(['status' => 'success', 'message' => 'Unit created successfully', 'data' => $unit], 201);
        } catch (ValidationException $e) {
            return response()->json(['status' => 'error', 'message' => $e->errors()], 422);
        } catch (QueryException $e) {
            return response()->json(['status' => 'error', 'message' => 'Database error while creating unit'], 500);
        } catch (Exception $e) {
            \Log::error('Error creating unit: ' . $e->getMessage());
            return response()->json(['status' => 'error', 'message' => 'An unexpected error occurred'], 500);
        }
    }

    // Get a specific unit
    public function show($id)
    {
        try {
            $unit = Unit::findOrFail($id);
            return response()->json(['status' => 'success', 'data' => $unit], 200);
        } catch (Exception $e) {
            return response()->json(['status' => 'error', 'message' => 'Unit not found'], 404);
        }
    }

    // Update a unit
    public function update(Request $request, $id)
    {
        try {
            $unit = Unit::findOrFail($id);

            $request->validate([
                'name' => 'required|string|max:255|unique:units,name,' . $id,
                'shortform' => 'required|string|max:10|unique:units,short_form,' . $id,
            ]);

            $unit->update($request->all());

            return response()->json(['status' => 'success', 'message' => 'Unit updated successfully', 'data' => $unit], 200);
        } catch (ValidationException $e) {
            return response()->json(['status' => 'error', 'message' => $e->errors()], 422);
        } catch (Exception $e) {
            return response()->json(['status' => 'error', 'message' => 'Unit not found or update failed'], 404);
        }
    }

    // Delete a unit
    public function destroy($id)
    {
        try {
            $unit = Unit::findOrFail($id);
            $unit->delete();
            return response()->json(['status' => 'success', 'message' => 'Unit deleted successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['status' => 'error', 'message' => 'Unit not found'], 404);
        }
    }
}
