<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Shift;
use Carbon\Carbon;
use Validator;

class ShiftController extends Controller
{
    /**
     * List all shifts with their assigned employees.
     */
    public function index(Request $request)
    {
        $shifts = Shift::with('employees')->get();
        return response()->json([
            'status' => 'success',
            'data'   => $shifts
        ]);
    }

    /**
     * Create a new shift.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'shift_code' => 'required|string|unique:shifts,shift_code',
            'start_time' => 'required|date_format:Y-m-d H:i:s',
            'end_time'   => 'required|date_format:Y-m-d H:i:s|after:start_time',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Validation error',
                'errors'  => $validator->errors()
            ], 422);
        }

        $shift = Shift::create($validator->validated());
        return response()->json([
            'status'  => 'success',
            'message' => 'Shift created successfully',
            'data'    => $shift
        ], 201);
    }

    /**
     * Show a specific shift.
     */
    public function show($id)
    {
        $shift = Shift::with('employees')->find($id);
        if (!$shift) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Shift not found'
            ], 404);
        }
        return response()->json([
            'status' => 'success',
            'data'   => $shift
        ]);
    }

    /**
     * Update a shift. Only allow update if the shift hasn't started.
     */
    public function update(Request $request, $id)
    {
        $shift = Shift::find($id);
        if (!$shift) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Shift not found'
            ], 404);
        }

        // Prevent updating a shift that has already started.
        if (Carbon::parse($shift->start_time)->isPast()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Cannot edit a shift that has already started'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'shift_code' => 'sometimes|required|string|unique:shifts,shift_code,' . $id,
            'start_time' => 'sometimes|required|date_format:Y-m-d H:i:s',
            'end_time'   => 'sometimes|required|date_format:Y-m-d H:i:s|after:start_time',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Validation error',
                'errors'  => $validator->errors()
            ], 422);
        }

        $shift->update($validator->validated());
        return response()->json([
            'status'  => 'success',
            'message' => 'Shift updated successfully',
            'data'    => $shift
        ]);
    }

    /**
     * Delete a shift. Only allow deletion if the shift hasn't started.
     */
    public function destroy($id)
    {
        $shift = Shift::find($id);
        if (!$shift) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Shift not found'
            ], 404);
        }
        if (Carbon::parse($shift->start_time)->isPast()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Cannot delete a shift that has already started'
            ], 403);
        }
        $shift->delete();
        return response()->json([
            'status'  => 'success',
            'message' => 'Shift deleted successfully'
        ]);
    }

    /**
     * Assign employees to a shift.
     * Expected input: { "employees": [1, 2, 3] } (IDs of employees)
     */
    public function assignEmployees(Request $request, $id)
    {
        $shift = Shift::find($id);
        if (!$shift) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Shift not found'
            ], 404);
        }

        $request->validate([
            'employees' => 'required|array',
            'employees.*' => 'exists:employees,id'
        ]);

        // Attach employees; if you need pivot data (e.g., clock in/out), adjust accordingly.
        $shift->employees()->sync($request->employees);

        return response()->json([
            'status'  => 'success',
            'message' => 'Employees assigned successfully',
            'data'    => $shift->load('employees')
        ]);
    }
}
