<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\ProductImage;
use Illuminate\Support\Str;
use DB;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ProductController extends Controller
{
    // ✅ Get All Products with Relationships
    public function index()
    {
        // Fetch products with relationships
        $products = Product::with(['category', 'subCategory', 'brand', 'unit', 'variants', 'images'])->get();

        return response()->json([
            'status' => 'success',
            'data' => $products->map(function ($product) {
                return [
                    'id'                => $product->id,
                    'name'              => $product->name,
                    'description'       => $product->description ?? '',
                    'sku'               => $product->sku ?? '',
                    'product_code'      => $product->product_code ?? '',
                    'category'          => $product->category?->name ?? 'Uncategorized',
                    'category_id'       => $product->category_id ?? null,
                    'sub_category'      => $product->subCategory?->name ?? null,
                    'sub_category_id'   => $product->sub_category_id ?? null,
                    'brand'             => $product->brand?->name ?? 'Unknown',
                    'unit'              => $product->unit?->name ?? 'Unknown',
                    'price'             => $product->price ?? 0,
                    'expiry_date'    => $product->expiry_date, // Add this line
                    'quantity'       => $product->quantity ?? 0,
                    'quantity_alert'    => $product->quantity_alert ?? 0,
                    'created_at'        => $product->created_at->format('Y-m-d'),
                    'images'            => $product->images->map(function ($image) {
                        return asset('storage/' . $image->image_path);
                    }),
                ];
            }),
        ]);
    }

    // ✅ Store a New Product
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name'              => 'required|string|max:255',
                'description'       => 'nullable|string',
                'category_id'       => 'required|exists:categories,id',
                'sub_category_id'   => 'nullable|exists:categories,id',
                'brand_id'          => 'nullable|exists:brands,id',
                'unit_id'           => 'nullable|exists:units,id',
                'quantity'          => 'required|integer|min:0',
                'price'             => 'required|numeric|min:0',
                'quantity_alert'    => 'nullable|integer|min:0',
                'variants'          => 'nullable|array',
                'images'            => 'nullable|array',
                'images.*'          => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            ]);

            DB::beginTransaction();

            // Format price
            $price = number_format((float) $request->price, 2, '.', '');

            // Handle nullable fields
            $subCategoryId = $request->filled('sub_category_id') ? $request->sub_category_id : null;
            $quantityAlert = $request->filled('quantity_alert') ? $request->quantity_alert : null;

            // Create product
            $product = new Product();
            $product->name              = $request->name;
            $product->description       = $request->description ?? null;
            $product->category_id       = $request->category_id;
            $product->sub_category_id   = $subCategoryId;
            $product->brand_id          = $request->brand_id ?? null;
            $product->unit_id           = $request->unit_id ?? null;
            $product->quantity          = $request->quantity;
            $product->price             = $price;
            $product->quantity_alert    = $quantityAlert;
            $product->product_code      = strtoupper(uniqid('PRD'));
            $product->sku               = strtoupper(substr($product->name, 0, 3)) . '-' . mt_rand(1000, 9999);
            $product->save();

            // Handle image upload
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $imagePath = $image->store('product_images', 'public');

                    ProductImage::create([
                        'product_id' => $product->id,
                        'image_path' => $imagePath,
                    ]);
                }
            }

            // Reload product with relationships
            $product->load('category', 'subCategory', 'brand', 'unit', 'images', 'variants');

            DB::commit();

            return response()->json([
                'status'  => 'success',
                'message' => 'Product created successfully.',
                'data'    => $product,
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status'  => 'error',
                'message' => 'Failed to create product',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    // ✅ Get a Single Product with Relationships
    public function show($id)
    {
        $product = Product::with(['category', 'subCategory', 'brand', 'units', 'variants', 'images'])->find($id);

        if (!$product) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Product not found',
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'data'   => $product,
        ]);
    }

    // ✅ Update Product
    public function update(Request $request, $id)
    {
        try {
            $product = Product::find($id);

            if (!$product) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Product not found',
                ], 404);
            }

            $request->validate([
                'name'              => 'sometimes|string|max:255',
                'description'       => 'nullable|string',
                'category_id'       => 'sometimes|exists:categories,id',
                'sub_category_id'   => 'nullable|exists:categories,id',
                'brand_id'          => 'nullable|exists:brands,id',
                'quantity'          => 'sometimes|integer|min:0',
                'price'             => 'sometimes|numeric|min:0',
                'quantity_alert'    => 'nullable|integer|min:0',
                'units'             => 'nullable|array',
                'units.*'           => 'exists:units,id',
                'variants'          => 'nullable|array',
                'variants.*.id'     => 'exists:variants,id',
                'variants.*.price'  => 'required|numeric|min:0',
                'variants.*.quantity' => 'required|integer|min:0',
                'variants.*.sku'    => 'required|string',
                'images'            => 'nullable|array',
                'images.*'          => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            ]);

            DB::beginTransaction();

            // Update product info
            $product->update($request->except(['units', 'variants', 'images']));

            // Update units if provided
            if ($request->has('units')) {
                $product->units()->sync($request->units);
            }

            // Update variants if provided
            if ($request->has('variants')) {
                $variantsData = [];
                foreach ($request->variants as $variant) {
                    $variantsData[$variant['id']] = [
                        'price'    => $variant['price'],
                        'quantity' => $variant['quantity'],
                        'sku'      => $variant['sku']
                    ];
                }
                $product->variants()->sync($variantsData);
            }

            // Update images if provided
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $imagePath = $image->store('product_images', 'public');
                    ProductImage::create([
                        'product_id' => $product->id,
                        'image_path' => $imagePath,
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'status'  => 'success',
                'message' => 'Product updated successfully',
                'data'    => $product->load('variants', 'brand', 'units', 'category', 'subCategory', 'images'),
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status'  => 'error',
                'message' => 'Failed to update product',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    // ✅ Batch Store Products from an Excel/CSV File
    public function batchStore(Request $request)
    {
        try {
            $productsData = $request->validate([
                'products' => 'required|array',
                'products.*.name' => 'required|string|max:255',
                'products.*.description' => 'nullable|string',
                'products.*.category_id' => 'required|exists:categories,id',
                'products.*.sub_category_id' => 'nullable|exists:categories,id',
                'products.*.brand_id' => 'nullable|exists:brands,id',
                'products.*.unit_id' => 'nullable|exists:units,id',
                'products.*.quantity' => 'required|integer|min:0',
                'products.*.price' => 'required|numeric|min:0',
                'products.*.tax_type' => 'required|string',
                'products.*.discount_type' => 'nullable|string',
                'products.*.discount_value' => 'nullable|numeric|min:0',
                'products.*.quantity_alert' => 'nullable|integer|min:0',
                'products.*.manufacturer_date' => 'required|date',
                'products.*.expiry_date' => 'required|date|after_or_equal:products.*.manufacturer_date',
                'products.*.images' => 'nullable|array',
                'products.*.images.*' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            ]);

            DB::beginTransaction();


            $products = [];
            foreach ($productsData['products'] as $productData) {
                // Ensure numeric values are correctly cast and rounded to two decimals.
                // Using round() returns a float.
                $productData['price'] = round((float)$productData['price'], 2);
                $productData['quantity'] = (int)$productData['quantity'];
                $productData['quantity_alert'] = isset($productData['quantity_alert']) ? (int)$productData['quantity_alert'] : 0;
                $productData['discount_value'] = isset($productData['discount_value']) ? round((float)$productData['discount_value'], 2) : null;

                // Convert dates to Carbon instances.
                $productData['manufacturer_date'] = \Carbon\Carbon::parse($productData['manufacturer_date']);
                $productData['expiry_date'] = \Carbon\Carbon::parse($productData['expiry_date']);

                // Create Product.
                $product = Product::create([
                    'name'              => $productData['name'],
                    'description'       => $productData['description'],
                    'category_id'       => $productData['category_id'],
                    'sub_category_id'   => $productData['sub_category_id'],
                    'brand_id'          => $productData['brand_id'],
                    'unit_id'           => $productData['unit_id'],
                    'quantity'          => $productData['quantity'],
                    'price'             => $productData['price'], // Now a properly rounded float.
                    'tax_type'          => $productData['tax_type'],
                    'discount_type'     => $productData['discount_type'],
                    'discount_value'    => $productData['discount_value'],
                    'quantity_alert'    => $productData['quantity_alert'],
                    'manufacturer_date' => $productData['manufacturer_date'],
                    'expiry_date'       => $productData['expiry_date'],
                    'product_code'      => strtoupper(uniqid('PRD')),
                    'sku'               => strtoupper(substr($productData['name'], 0, 3)) . '-' . mt_rand(1000, 9999),
                ]);

                // Save Images if provided.
                if (!empty($productData['images']) && is_array($productData['images'])) {
                    foreach ($productData['images'] as $image) {
                        $imagePath = $image->store('product_images', 'public');
                        // Assuming you have a ProductImage model already imported
                        \App\Models\ProductImage::create([
                            'product_id' => $product->id,
                            'image_path' => $imagePath
                        ]);
                    }
                }

                $products[] = $product;
            }

            DB::commit();

            return response()->json([
                'status'  => 'success',
                'message' => 'Batch products created successfully.',
                'data'    => $products,
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status'  => 'error',
                'message' => 'Failed to create batch products',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }


//Return Expired Goods
public function expiringProducts()
{
    try {
        $today = now()->format('Y-m-d');
        $nextWeek = now()->addDays(7)->format('Y-m-d');

        $products = Product::where(function($query) use ($today, $nextWeek) {
            $query->whereBetween('expiry_date', [$today, $nextWeek])
                  ->orWhere('expiry_date', '<', $today);
        })
        ->where('expiry_date', '!=', '9999-12-31')
        ->with(['category', 'brand', 'images'])
        ->get()
        ->map(function ($product) use ($today) {
            return [
                'id' => $product->id,
                'name' => $product->name,
                'sku' => $product->sku,
                'product_code' => $product->product_code,
                'expiry_date' => $product->expiry_date->format('Y-m-d'),
                'status' => $product->expiry_date < $today ? 'expired' : 'expiring_soon',
                'category' => $product->category->name ?? 'N/A',
                'brand' => $product->brand->name ?? 'N/A',
                'images' => $product->images->map(fn($img) => asset('storage/'.$img->image_path))
            ];
        });

        return response()->json([
            'status' => 'success',
            'data' => $products // Direct array without nested 'data'
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'status' => 'error',
            'message' => 'Failed to fetch products',
            'error' => $e->getMessage()
        ], 500);
    }
}


}
