<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use PDF; // Ensure barryvdh/laravel-dompdf is installed and configured

class OrderController extends Controller
{
    public function index()
    {
        $orders = Order::with('orderItems.product')->get(); // eager load product relation

        return response()->json([
            'status' => 'success',
            'data' => $orders,
        ]);
    }

    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'transaction_id'   => 'required|string|unique:orders,transaction_id',
                'cart'             => 'required|array',
                'cart.*.id'        => 'required|exists:products,id',
                'cart.*.quantity'  => 'required|integer|min:1',
                'cart.*.price'     => 'required|numeric|min:0',
                'subtotal'         => 'required|numeric',
                'tax'              => 'required|numeric',
                'shipping'         => 'nullable|numeric',
                'discount'         => 'nullable|numeric',
                'total'            => 'required|numeric',
                'payment_method'   => 'required|string',
                'payment_details'  => 'nullable|array',
                'cashier'          => 'required|string',
                'status'           => 'required|string',
            ]);

            // Create the order record
            $order = Order::create([
                'order_number'   => strtoupper(uniqid('ORD-')),
                'transaction_id' => $validated['transaction_id'],
                'cashier'        => $validated['cashier'],
                'status'         => $validated['status'],
                'subtotal'       => $validated['subtotal'],
                'tax'            => $validated['tax'],
                'shipping'       => $validated['shipping'] ?? 0,
                'discount'       => $validated['discount'] ?? 0,
                'total_amount'   => $validated['total'],
                'payment_method' => $validated['payment_method'],
                'payment_details'=> $validated['payment_details'] ?? null,
            ]);

            // Process each cart item, update product stock, and create order items
            foreach ($validated['cart'] as $item) {
                $product = Product::find($item['id']);
                if ($product->quantity < $item['quantity']) {
                    \Log::error("Insufficient stock for product: {$product->name}. Available: {$product->quantity}, Requested: {$item['quantity']}");
                    return response()->json([
                        'status' => 'error',
                        'message' => "Insufficient stock for product: {$product->name}. Available: {$product->quantity}, Requested: {$item['quantity']}",
                    ], 400);
                }
                // Update product stock
                $product->update([
                    'quantity' => $product->quantity - $item['quantity'],
                ]);
                // Create order item
                OrderItem::create([
                    'order_id'   => $order->id,
                    'product_id' => $item['id'],
                    'quantity'   => $item['quantity'],
                    'price'      => $item['price'],
                    'total'      => $item['price'] * $item['quantity'],
                ]);
            }

            // Generate the PDF receipt using a Blade view
            $order = $order->load('orderItems.product'); // eager load the product relation for each order item
            $pdf = PDF::loadView('orders.receipt', ['order' => $order]);
            $receiptPath = 'receipts/receipt_' . $order->order_number . '.pdf';
            Storage::put('public/' . $receiptPath, $pdf->output());

            // Save the receipt path in the order record
            $order->update(['receipt_path' => $receiptPath]);

            return response()->json([
                'status' => 'success',
                'message' => 'Order processed successfully.',
                'data' => $order,
            ], 201);
        } catch (\Exception $e) {
            \Log::error('Error in OrderController@store: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred while processing the order.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function show($id)
    {
        $order = Order::with('orderItems.product')->find($id);

        if (!$order) {
            return response()->json([
                'status' => 'error',
                'message' => 'Order not found.',
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'data' => $order,
        ]);
    }

    public function update(Request $request, $id)
    {
        $order = Order::find($id);

        if (!$order) {
            return response()->json([
                'status' => 'error',
                'message' => 'Order not found.',
            ], 404);
        }

        $order->update($request->all());

        return response()->json([
            'status' => 'success',
            'message' => 'Order updated successfully.',
            'data' => $order,
        ]);
    }

    public function destroy($id)
    {
        $order = Order::find($id);

        if (!$order) {
            return response()->json([
                'status' => 'error',
                'message' => 'Order not found.',
            ], 404);
        }

        $order->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Order deleted successfully.',
        ]);
    }
}
