<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Employee;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Validator;
use DB;

class EmployeeController extends Controller
{
    /**
     * Display a listing of employees with their relationships.
     */
    public function index()
    {
        $employees = Employee::with(['user', 'department', 'designation', 'shift'])->get();
        return response()->json([
            'status' => 'success',
            'data'   => $employees
        ]);
    }

/**
 * Store a newly created employee and user account.
 */
public function store(Request $request)
{
    $validator = Validator::make($request->all(), [
        'name'             => 'required|string|max:255',
        'role'             => 'required|string', // Expected: Employee, Manager, or Owner
        'designation_id'   => 'required|exists:designations,id',
        'department_id'    => 'required|exists:departments,id',
        'employee_id'      => 'required|string|unique:employees,employee_id',
        'email'            => 'required|email|unique:employees,email|unique:users,email',
        'phone'            => 'required|string',
        'shift_id'         => 'nullable|exists:shifts,id',
        'status'           => 'required|in:Probation,Pending,Active,Suspended,Terminated',
        'password'         => 'required|string|min:6',
        'marital_status'   => 'nullable|string',
        'gender'           => 'nullable|string',
        'national_id'      => 'nullable|string',
        'kra_pin'          => 'nullable|string',
        'shif_no'          => 'nullable|string',
        'nssf_no'          => 'nullable|string',
        'bank_name'        => 'nullable|string',
        'account_no'       => 'nullable|string',
        'salary'           => 'nullable|numeric',
        'allowances'       => 'nullable|string',
        'deductions'       => 'nullable|string',
        'passport_image'   => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
    ]);

    if ($validator->fails()) {
        return response()->json([
            'status'  => 'error',
            'message' => 'Validation error',
            'errors'  => $validator->errors()
        ], 422);
    }

    DB::beginTransaction();
    try {
        // Create the user account for login.
        $user = User::create([
            'name'     => $request->name,
            'email'    => $request->email,
            'password' => Hash::make($request->password),
            'role_id'  => $this->getRoleIdFromRole($request->role),
        ]);

        // Handle passport image upload if provided.
        $passportImagePath = null;
        if ($request->hasFile('passport_image')) {
            $file = $request->file('passport_image');
            // Check that the file is valid and has a non-zero size.
            if ($file->isValid() && $file->getSize() > 0) {
                $passportImagePath = $file->store('employee_passports', 'public');
            } else {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'The passport image failed to upload.',
                    'errors'  => ['passport_image' => ['The passport image failed to upload.']]
                ], 422);
            }
        }

        // Create the employee record.
        $employee = Employee::create([
            'name'             => $request->name,
            'role'             => $request->role,
            'designation_id'   => $request->designation_id,
            'department_id'    => $request->department_id,
            'employee_id'      => $request->employee_id,
            'email'            => $request->email,
            'phone'            => $request->phone,
            'shift_id'         => $request->shift_id, // nullable field
            'status'           => $request->status,
            'user_id'          => $user->id,
            'marital_status'   => $request->marital_status ?: null,
            'gender'           => $request->gender ?: null,
            'national_id'      => $request->national_id ?: null,
            'kra_pin'          => $request->kra_pin ?: null,
            'shif_no'          => $request->shif_no ?: null,
            'nssf_no'          => $request->nssf_no ?: null,
            'bank_name'        => $request->bank_name ?: null,
            'account_no'       => $request->account_no ?: null,
            'salary'           => $request->salary,
            'allowances'       => $request->allowances,
            'deductions'       => $request->deductions,
            'passport_image'   => $passportImagePath,
        ]);

        DB::commit();

        return response()->json([
            'status'  => 'success',
            'message' => 'Employee and user account created successfully',
            'data'    => $employee
        ], 201);
    } catch (\Exception $e) {
        DB::rollBack();
        return response()->json([
            'status'  => 'error',
            'message' => 'Failed to create employee',
            'error'   => $e->getMessage(),
        ], 500);
    }
}

/**
 * Helper method to map role names to role IDs.
 */
private function getRoleIdFromRole($role)
{
    $roles = [
        'Employee' => 2,
        'Manager'  => 1,
        'Owner'    => 3,
    ];

    return $roles[$role] ?? 2;
}


    /**
     * Display the specified employee with relationships.
     */
    public function show($id)
    {
        $employee = Employee::with(['user', 'department', 'designation', 'shift'])->find($id);
        if (!$employee) {
            return response()->json(['message' => 'Employee not found'], 404);
        }
        return response()->json($employee);
    }

    /**
     * Update the specified employee.
     */
    public function update(Request $request, $id)
    {
        $employee = Employee::find($id);
        if (!$employee) {
            return response()->json(['message' => 'Employee not found'], 404);
        }

        $data = $request->all();
        // Normalize empty strings to null for nullable fields.
        $nullableFields = ['shift_id', 'marital_status', 'gender', 'national_id', 'kra_pin', 'shif_no', 'nssf_no', 'bank_name', 'account_no', 'salary', 'allowances', 'deductions'];
        foreach ($nullableFields as $field) {
            if (isset($data[$field]) && $data[$field] === '') {
                $data[$field] = null;
            }
        }

        $validator = Validator::make($data, [
            'name'             => 'sometimes|required|string|max:255',
            'role'             => 'sometimes|required|string',
            'designation_id'   => 'sometimes|required|exists:designations,id',
            'department_id'    => 'sometimes|required|exists:departments,id',
            'email'            => 'sometimes|required|email|unique:employees,email,' . $id,
            'phone'            => 'sometimes|required|string',
            'shift_id'         => 'nullable|exists:shifts,id',
            'status'           => 'sometimes|required|in:Probation,Pending,Active,Suspended,Terminated',
            'marital_status'   => 'nullable|string',
            'gender'           => 'nullable|string',
            'national_id'      => 'nullable|string',
            'kra_pin'          => 'nullable|string',
            'shif_no'          => 'nullable|string',
            'nssf_no'          => 'nullable|string',
            'bank_name'        => 'nullable|string',
            'account_no'       => 'nullable|string',
            'salary'           => 'nullable|numeric',
            'allowances'       => 'nullable|string',
            'deductions'       => 'nullable|string',
            'passport_image'   => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation error',
                'errors'  => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            // Update passport image if provided and valid.
            if ($request->hasFile('passport_image') && $request->file('passport_image')->isValid()) {
                $passportImagePath = $request->file('passport_image')->store('employee_passports', 'public');
                $employee->passport_image = $passportImagePath;
            }

            $employee->update($request->except('passport_image'));
            DB::commit();
            return response()->json([
                'message' => 'Employee updated successfully',
                'data'    => $employee->load(['user', 'department', 'designation', 'shift']),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Failed to update employee',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified employee.
     */
    public function destroy($id)
    {
        $employee = Employee::find($id);
        if (!$employee) {
            return response()->json(['message' => 'Employee not found'], 404);
        }
        $employee->delete();
        return response()->json(['message' => 'Employee deleted successfully']);
    }

}
