<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Designation;
use Validator;

class DesignationController extends Controller
{
    /**
     * Display a listing of designations.
     */
    public function index()
    {
        $designations = Designation::all();
        return response()->json([
            'status' => 'success',
            'data'   => $designations
        ]);
    }

    /**
     * Store a newly created designation.
     */
    public function store(Request $request)
    {
        // Validate the input. Note: max_salary must be greater than or equal to min_salary.
        $validator = Validator::make($request->all(), [
            'title'         => 'required|string|unique:designations,title|max:255',
            'job_group'     => 'nullable|string|max:255',
            'min_salary'    => 'required|numeric|min:0',
            'max_salary'    => 'required|numeric|min:0|gte:min_salary',
            'benefits'      => 'nullable|string',    // You may store JSON here if needed.
            'deductions'    => 'nullable|string',    // E.g., a JSON object containing tax bracket details.
            'allowances'    => 'nullable|string',    // E.g., a JSON object listing allowances.
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Validation error',
                'errors'  => $validator->errors()
            ], 422);
        }

        $designation = Designation::create($validator->validated());

        return response()->json([
            'status'  => 'success',
            'message' => 'Designation created successfully',
            'data'    => $designation
        ], 201);
    }

    /**
     * Display the specified designation.
     */
    public function show($id)
    {
        $designation = Designation::find($id);
        if (!$designation) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Designation not found'
            ], 404);
        }
        return response()->json([
            'status' => 'success',
            'data'   => $designation
        ]);
    }

    /**
     * Update the specified designation.
     */
    public function update(Request $request, $id)
    {
        $designation = Designation::find($id);
        if (!$designation) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Designation not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'title'         => "sometimes|required|string|unique:designations,title,{$id}|max:255",
            'job_group'     => 'nullable|string|max:255',
            'min_salary'    => 'sometimes|required|numeric|min:0',
            'max_salary'    => 'sometimes|required|numeric|min:0|gte:min_salary',
            'benefits'      => 'nullable|string',
            'deductions'    => 'nullable|string',
            'allowances'    => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Validation error',
                'errors'  => $validator->errors()
            ], 422);
        }

        $designation->update($validator->validated());

        return response()->json([
            'status'  => 'success',
            'message' => 'Designation updated successfully',
            'data'    => $designation
        ]);
    }

    /**
     * Remove the specified designation.
     */
    public function destroy($id)
    {
        $designation = Designation::find($id);
        if (!$designation) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Designation not found'
            ], 404);
        }
        $designation->delete();
        return response()->json([
            'status'  => 'success',
            'message' => 'Designation deleted successfully'
        ]);
    }
}
