<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Department;
use Validator;

class DepartmentController extends Controller
{
    /**
     * List all departments with their head (even if null).
     */
    public function index()
    {
        // Fetch all departments including their head relationship.
        $departments = Department::with('head')->get();

        // Map each department to always include the "head" key.
        $mappedDepartments = $departments->map(function ($department) {
            return [
                'id'             => $department->id,
                'name'           => $department->name,
                'description'    => $department->description,
                'head'           => $department->head, // This will be null if no head is assigned.
                'created_at'     => $department->created_at->format('Y-m-d'),
                // You can add more fields as needed...
            ];
        });

        return response()->json([
            'status' => 'success',
            'data'   => $mappedDepartments
        ]);
    }

    /**
     * Store a new department.
     */
    public function store(Request $request)
    {
        // Validate the input. If head_id is provided, ensure it exists in the employees table.
        $validator = Validator::make($request->all(), [
            'name'        => 'required|string|unique:departments,name|max:255',
            'description' => 'nullable|string',
            'head_id'     => 'nullable|exists:employees,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Validation error',
                'errors'  => $validator->errors()
            ], 422);
        }

        // Create the department with validated data.
        $department = Department::create($validator->validated());

        // Load the head relationship (will be null if head_id not provided)
        $department->load('head');

        return response()->json([
            'status'  => 'success',
            'message' => 'Department created successfully',
            'data'    => [
                'id'          => $department->id,
                'name'        => $department->name,
                'description' => $department->description,
                'head'        => $department->head, // returns null if not assigned
                'created_at'  => $department->created_at->format('Y-m-d'),
            ]
        ], 201);
    }

    /**
     * Show a single department with its head.
     */
    public function show($id)
    {
        $department = Department::with('head')->find($id);
        if (!$department) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Department not found'
            ], 404);
        }
        return response()->json([
            'status' => 'success',
            'data'   => [
                'id'          => $department->id,
                'name'        => $department->name,
                'description' => $department->description,
                'head'        => $department->head, // This will be null if not assigned.
                'created_at'  => $department->created_at->format('Y-m-d'),
            ]
        ]);
    }

    /**
     * Update an existing department.
     */
    public function update(Request $request, $id)
    {
        $department = Department::find($id);
        if (!$department) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Department not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'name'        => "sometimes|required|string|unique:departments,name,{$id}|max:255",
            'description' => 'nullable|string',
            'head_id'     => 'nullable|exists:employees,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Validation error',
                'errors'  => $validator->errors()
            ], 422);
        }

        $department->update($validator->validated());

        // Reload the head relationship.
        $department->load('head');

        return response()->json([
            'status'  => 'success',
            'message' => 'Department updated successfully',
            'data'    => [
                'id'          => $department->id,
                'name'        => $department->name,
                'description' => $department->description,
                'head'        => $department->head,
                'created_at'  => $department->created_at->format('Y-m-d'),
            ]
        ]);
    }

    /**
     * Delete a department.
     */
    public function destroy($id)
    {
        $department = Department::find($id);
        if (!$department) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Department not found'
            ], 404);
        }
        $department->delete();
        return response()->json([
            'status'  => 'success',
            'message' => 'Department deleted successfully'
        ]);
    }
}
