<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use App\Models\Category;
use App\Models\Product;

class CategoryController extends Controller
{
    // ✅ Fetch all top-level categories (No Parent)
    public function index()
    {
        // Fetch only parent categories (excluding subcategories)
        $categories = Category::whereNull('parent_id')
            ->with('subcategories', 'products') // Include subcategories and products
            ->get();

        // Format the response to include product counts, including subcategories
        $categories->transform(function ($category) {
            return [
                'id' => $category->id,
                'name' => $category->name,
                'description' => $category->description,
                'image' => $category->image ? asset("storage/{$category->image}") : '/images/pos-placeholder.jpg',
                'products_count' => $this->getTotalProductsInCategory($category),
            ];
        });

        return response()->json([
            'status' => 'success',
            'data' => $categories,
        ]);
    }

    // ✅ Fetch all products in a category (including its subcategories)
    public function show($id)
    {
        $category = Category::with('subcategories')->find($id);

        if (!$category) {
            return response()->json([
                'status' => 'error',
                'message' => 'Category not found',
            ], 404);
        }

        // Fetch products directly under this category and all its subcategories
        $categoryIds = $this->getCategoryWithChildrenIds($category);

        $products = Product::whereIn('category_id', $categoryIds)
            ->with('images') // Ensure product images are included
            ->get();

        return response()->json([
            'status' => 'success',
            'data' => [
                'category' => $category,
                'products' => $products
            ],
        ]);
    }

    // ✅ Store a new category
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|unique:categories,name|max:255',
                'description' => 'nullable|string',
                'parent_id' => 'nullable|exists:categories,id',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            ]);

            if (Category::where('name', $request->name)->exists()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Category name already exists.',
                ], 400);
            }

            $imagePath = null;
            if ($request->hasFile('image')) {
                $imagePath = $request->file('image')->store('category_images', 'public');
            }

            $category = Category::create([
                'name' => $request->name,
                'description' => $request->description,
                'image' => $imagePath,
                'parent_id' => $request->parent_id ?? null,
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Category created successfully',
                'data' => $category->load('subcategories'),
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to create category',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    // ✅ Helper function to get all product counts including subcategories
    private function getTotalProductsInCategory($category)
    {
        // Fetch products in the current category and all its subcategories
        $categoryIds = $this->getCategoryWithChildrenIds($category);

        return Product::whereIn('category_id', $categoryIds)->count();
    }

    // ✅ Helper function to get category ID including children categories
    private function getCategoryWithChildrenIds($category)
    {
        $categoryIds = [$category->id]; // Start with the main category

        foreach ($category->subcategories as $subcategory) {
            $categoryIds = array_merge($categoryIds, $this->getCategoryWithChildrenIds($subcategory));
        }

        return $categoryIds;
    }
}
