<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Brand;

class BrandController extends Controller
{
    // Fetch all brands
    public function index()
    {
        $brands = Brand::all();
        return response()->json(['status' => 'success', 'data' => $brands]);
    }

    // Create a new brand
    public function store(Request $request)
{
    try {
        $request->validate([
            'name' => 'required|string|max:255',
            'manufacturer' => 'nullable|string|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048', // Validate logo
        ]);

        // Handle logo upload
        $logoPath = null;
        if ($request->hasFile('logo')) {
            $logoPath = $request->file('logo')->store('brand_logos', 'public');
        }

        $brand = Brand::create([
            'name' => $request->name,
            'manufacturer' => $request->manufacturer,
            'logo' => $logoPath, // Save logo path
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Brand created successfully.',
            'data' => $brand,
        ], 201);
    } catch (\Exception $e) {
        return response()->json([
            'status' => 'error',
            'message' => 'Failed to create brand.',
            'error' => $e->getMessage(),
        ], 500);
    }
}


    // Get a specific brand
    public function show($id)
    {
        $brand = Brand::find($id);
        if (!$brand) {
            return response()->json(['status' => 'error', 'message' => 'Brand not found'], 404);
        }
        return response()->json(['status' => 'success', 'data' => $brand]);
    }

    // Update a brand
    public function update(Request $request, $id)
{
    $brand = Brand::find($id);
    if (!$brand) {
        return response()->json([
            'status' => 'error',
            'message' => 'Brand not found',
        ], 404);
    }

    $request->validate([
        'name' => 'sometimes|string|max:255',
        'manufacturer' => 'nullable|string|max:255',
        'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
    ]);

    // Handle new logo upload
    if ($request->hasFile('logo')) {
        if ($brand->logo) {
            Storage::disk('public')->delete($brand->logo); // Delete old logo
        }
        $brand->logo = $request->file('logo')->store('brand_logos', 'public');
    }

    $brand->update($request->except(['logo']));

    return response()->json([
        'status' => 'success',
        'message' => 'Brand updated successfully.',
        'data' => $brand,
    ]);
}


    // Delete a brand
    public function destroy($id)
    {
        $brand = Brand::find($id);
        if (!$brand) {
            return response()->json(['status' => 'error', 'message' => 'Brand not found'], 404);
        }

        $brand->delete();
        return response()->json(['status' => 'success', 'message' => 'Brand deleted successfully']);
    }
}
