<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use App\Models\User;

class AuthController extends Controller
{
    /**
     * Register a new user.
     */
    public function register(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|email|unique:users,email',
                'password' => 'required|string|min:6|confirmed',
            ]);

            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
            ]);

            Log::channel('pos')->info('User registered successfully', [
                'email' => $user->email,
                'time' => now(),
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'User registered successfully',
            ], 201);
        } catch (\Exception $e) {
            Log::channel('pos')->error('User registration failed', [
                'error' => $e->getMessage(),
                'time' => now(),
            ]);

            return response()->json([
                'status' => 'error',
                'message' => 'Registration failed',
            ], 500);
        }
    }

    /**
     * Login a user and generate access/refresh tokens.
     */
    public function login(Request $request)
    {
        try {
            $request->validate([
                'email' => 'required|email',
                'password' => 'required|string',
            ]);

            $user = User::where('email', $request->email)->first();

            if (!$user || !Hash::check($request->password, $user->password)) {
                Log::channel('pos')->error('Login failed: Invalid credentials', [
                    'email' => $request->email,
                    'time' => now(),
                ]);

                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid credentials',
                ], 401);
            }

            $accessToken = $user->createToken('auth_token')->plainTextToken;
            $refreshToken = base64_encode(Str::random(40));
            $user->update(['refresh_token' => $refreshToken]);

            Log::channel('pos')->info('Login successful', [
                'user_id' => $user->id,
                'time' => now(),
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Login successful',
                'data' => [
                    'accessToken' => $accessToken,
                    'refreshToken' => $refreshToken,
                    'tokenType' => 'Bearer',
                    'expiresIn' => 3600,
                ],
            ]);
        } catch (\Exception $e) {
            Log::channel('pos')->error('Login error', [
                'error' => $e->getMessage(),
                'time' => now(),
            ]);

            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred during login',
            ], 500);
        }
    }

    /**
     * Refresh an expired access token.
     */
    public function refreshToken(Request $request)
    {
        try {
            $request->validate([
                'refreshToken' => 'required|string',
            ]);

            $user = User::where('refresh_token', $request->refreshToken)->first();

            if (!$user) {
                Log::channel('pos')->error('Refresh token invalid', [
                    'refreshToken' => $request->refreshToken,
                    'time' => now(),
                ]);

                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid refresh token',
                ], 401);
            }

            $updatedAt = $user->updated_at;
            if (now()->diffInHours($updatedAt) > 24) {
                Log::channel('pos')->error('Refresh token expired', [
                    'user_id' => $user->id,
                    'time' => now(),
                ]);

                return response()->json([
                    'status' => 'error',
                    'message' => 'Refresh token expired',
                ], 401);
            }

            $accessToken = $user->createToken('auth_token')->plainTextToken;
            $user->touch();

            Log::channel('pos')->info('Access token refreshed', [
                'user_id' => $user->id,
                'time' => now(),
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Access token refreshed',
                'data' => [
                    'accessToken' => $accessToken,
                    'tokenType' => 'Bearer',
                    'expiresIn' => 3600,
                ],
            ]);
        } catch (\Exception $e) {
            Log::channel('pos')->error('Token refresh error', [
                'error' => $e->getMessage(),
                'time' => now(),
            ]);

            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred during token refresh',
            ], 500);
        }
    }

    /**
     * Logout the user and invalidate tokens.
     */
    public function logout(Request $request)
    {
        try {
            $user = Auth::user();
            $user->tokens()->delete();
            $user->update(['refresh_token' => null]);

            Log::channel('pos')->info('User logged out successfully', [
                'user_id' => $user->id,
                'time' => now(),
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Logged out successfully',
            ]);
        } catch (\Exception $e) {
            Log::channel('pos')->error('Logout error', [
                'error' => $e->getMessage(),
                'time' => now(),
            ]);

            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred during logout',
            ], 500);
        }
    }
}
