<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Allowance;
use Illuminate\Support\Facades\Validator;

class AllowanceController extends Controller
{
    /**
     * Display a listing of all allowances.
     */
    public function index()
    {
        $allowances = Allowance::all();
        return response()->json([
            'status' => 'success',
            'data'   => $allowances
        ]);
    }

    /**
     * Store a newly created allowance.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name'        => 'required|string|unique:allowances,name|max:255',
            'description' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Validation error',
                'errors'  => $validator->errors()
            ], 422);
        }

        $allowance = Allowance::create($validator->validated());

        return response()->json([
            'status'  => 'success',
            'message' => 'Allowance created successfully',
            'data'    => $allowance
        ], 201);
    }

    /**
     * Display the specified allowance.
     */
    public function show($id)
    {
        $allowance = Allowance::find($id);
        if (!$allowance) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Allowance not found'
            ], 404);
        }
        return response()->json([
            'status' => 'success',
            'data'   => $allowance
        ]);
    }

    /**
     * Update the specified allowance.
     */
    public function update(Request $request, $id)
    {
        $allowance = Allowance::find($id);
        if (!$allowance) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Allowance not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'name'        => "sometimes|required|string|unique:allowances,name,{$id}|max:255",
            'description' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Validation error',
                'errors'  => $validator->errors()
            ], 422);
        }

        $allowance->update($validator->validated());

        return response()->json([
            'status'  => 'success',
            'message' => 'Allowance updated successfully',
            'data'    => $allowance
        ]);
    }

    /**
     * Remove the specified allowance.
     */
    public function destroy($id)
    {
        $allowance = Allowance::find($id);
        if (!$allowance) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Allowance not found'
            ], 404);
        }
        $allowance->delete();
        return response()->json([
            'status'  => 'success',
            'message' => 'Allowance deleted successfully'
        ]);
    }
}
